/*
 * © Vidéotron, 2017
 * dccAuthenticationUtil.js - consulter l'équipe front-end pour info.
 */
(function(window, document, undefined) {
    'use strict';

    var state = {
        init: false,
        callOnReadyWhenInit: false,
    	isLoginBusy: false,
    };

    function getIsDebug() {
        var queryString = window.location.search.slice(1);
        if (!queryString) {
            return false;
        }
        return queryString.indexOf('vl-sso-debug=true') !== -1;
    }

    var opts = {
        dccHost: '',
        authLink: '/oam/server/authentication',
        successUrl: '/vl-sso-bin/login-app-result.pl', // note that this must be a valid location, but cannot be customised.
        lang: 'fr',
        clearPasswordInput: true,
		optionalFormKeys: [],
        debug: getIsDebug()
    };

    function init(dccOptionsDescriptor) {
        var props = [
            'dccHost',
            'onDccResponse',
            'onReady',
            'loginFormId',
            'lang',
            'onIsAnonymous',
            'onDccBefore',
            'pingUrl',
            'clearPasswordInput',
			'optionalFormKeys'
        ];

        if (typeof dccOptionsDescriptor !== 'object') {
            console.log('dccAuthenticationUtil.js: ERROR -> init() must receive a valid dccOptions descriptor');
            return;
        }

        for (var i = 0; i < props.length; i++) {
            var prop = props[i];
            if (dccOptionsDescriptor[prop]) {
                opts[prop] = dccOptionsDescriptor[prop];
            }
        }
        state.init = true;
        getLoginStatus();
        if (state.callOnReadyWhenInit) {
        	dccOnReady(null);
        }
    }

    function traceLog(msg) {
        if (!opts.debug) return;
        console.log('TRACE: ' + msg);
    }

    function submitToDcc(formElement) {
        traceLog('dccAuthenticationUtil::submitToDcc() -> entering');

        if (opts.onDccBefore && typeof opts.onDccBefore === 'function') {
            opts.onDccBefore();
        }

        if (opts.onIsAnonymous && typeof opts.onIsAnonymous === 'function' && opts.onIsAnonymous()) {
            _doGetMethodPing(function(message) {
                state.isLoginBusy = false;
                var loginFormElm = document.getElementById(opts.loginFormId);
                loginFormElm.submit();
            });
            return true;
        }

        var authLink = opts.dccHost + opts.authLink;
        var xhr = new XMLHttpRequest();
        var i,
            j,
            callbackCalled = false;

        function callbackWrapper(callingContext) {
            if (opts.onDccResponse && !callbackCalled) {
                callbackCalled = true;
                traceLog('dccAuthenticationUtil::submitToDcc() -> opts.onDccResponse(' + callingContext + ')');
                if (opts.clearPasswordInput) {
                    var passwordElm = document.getElementById(opts.loginFormId).elements['password'];
                    if (passwordElm && (!response || response.status !== 'success')) {
                        passwordElm.value = '';
                    }
                }
                opts.onDccResponse(xhr.status, response);
            } else {
                console.log('dccAuthenticationUtil::submitToDcc() -> ERROR: onDccResponse() is invalid (' + callingContext + ')');
            }
            traceLog('dccAuthenticationUtil::submitToDcc() -> isLoginBusy=false - ' + callingContext);
            state.isLoginBusy = false;
        }

        var response;

        function _pingLoginResult(callback) {
            var pingLoginResultXhr = new XMLHttpRequest();
            var pingLoginResultUrl = opts.dccHost + opts.successUrl;

            pingLoginResultXhr.open('get', pingLoginResultUrl, true);
            pingLoginResultXhr.withCredentials = true;
            pingLoginResultXhr.send(null);

            pingLoginResultXhr.onreadystatechange = function() {
                if (pingLoginResultXhr.readyState !== 4) {
                    return;
                }
                if (pingLoginResultXhr.status !== 200 && pingLoginResultXhr.status !== 201 && pingLoginResultXhr.status !== 304) {
                    traceLog(
                        'dccAuthenticationUtil::submitToDcc()::pingLoginResult() -> pre-ping DCC workaround, invalid http status: status=' +
                            pingLoginResultXhr.status +
                            ', statusText=' +
                            pingLoginResultXhr.statusText
                    );
                }

                if (pingLoginResultXhr.status > 0) {
                    callback();
                }
            };
            pingLoginResultXhr.onerror = function() {
                traceLog(
                    'dccAuthenticationUtil::submitToDcc()::pingLoginResult() -> ping DCC workaround, XHR Auth error: status=' +
                        pingLoginResultXhr.status +
                        ', statusText=' +
                        pingLoginResultXhr.statusText
                );
                callback();
            };
        }

        function _doGetMethodPing(doPingCallback) {
            var xhrPing = new XMLHttpRequest();

            if (opts.pingUrl) {
                xhrPing.open('get', opts.pingUrl, true);
                xhrPing.withCredentials = true;
                xhrPing.send(null);
            } else {
                // call the callback when the pingUrl is not specified.
                doPingCallback('pingUrl not specified');
            }

            xhrPing.onreadystatechange = function() {
                if (xhrPing.readyState !== 4) {
                    return;
                }
                if (xhrPing.status !== 200 && xhrPing.status !== 201 && xhrPing.status !== 304) {
                    traceLog(
                        'dccAuthenticationUtil::submitToDcc() -> ping DCC workaround, invalid http status: status=' +
                            xhrPing.status +
                            ', statusText=' +
                            xhrPing.statusText
                    );
                }
                // when there is an error with the ping, do nothing as the onerror handler will try the ping with an iframe.
                // the _pingWithFrame method will call the callback.
                if (xhrPing.status > 0) {
                    doPingCallback('pingUrl specified');
                }
            };
            xhrPing.onerror = function() {
                traceLog(
                    'dccAuthenticationUtil::submitToDcc() -> ping DCC workaround, XHR Auth error: status=' +
                        xhrPing.status +
                        ', statusText=' +
                        xhrPing.statusText
                );
                // call the callback whether or not there is an error with the pingUrl (when specified).
                _pingLoginResult(function() {
                    _pingWithFrame(doPingCallback);
                });
            };
        }

        function _pingWithFrame(callback) {
            var timeoutDuration = 3 * 60 * 1000; // 3 minutes timeout
            var timeout = null;
            var iframeId = 'authPingFrame' + Math.round(Math.random() * 1000000);
            var iframe = document.createElement('iframe');

            iframe.src = opts.pingUrl;
            iframe.id = iframeId;
            iframe.style.width = 0;
            iframe.style.height = 0;
            iframe.style.border = 0;

            var frameLoaded = function() {
                clearFrame();
                clearTimeout(timeout);
                timeout = null;
                traceLog('dccAuthenticationUtil::submitToDcc() -> ping DCC with iframe workaround, iframe load event fired.');
                callback('ping with iframe success');
            };

            var clearFrame = function() {
                var exisitingFrame = document.getElementById(iframeId);
                if (exisitingFrame) {
                    exisitingFrame.parentNode.removeChild(exisitingFrame);
                }
            };

            // in case the iframe fails to fire load event, probably means it in
            // error state..
            var timeout = setTimeout(function() {
                iframe.onload = null;
                clearFrame();
                traceLog(
                    'dccAuthenticationUtil::submitToDcc() -> ping DCC with iframe workaround, iframe load event not fired, timed out.'
                );
                callback('ping with iframe failed, timed out');
            }, timeoutDuration);

            document.body.appendChild(iframe);
            iframe.onload = frameLoaded;
        }

        xhr.onreadystatechange = function() {
            if (xhr.readyState !== 4) {
                return;
            }
            traceLog(
                'dccAuthenticationUtil::submitToDcc()::onreadystatechange -> XHR Auth status - ' +
                    authLink +
                    ': ' +
                    xhr.status +
                    ' ; ' +
                    xhr.statusText
            );
            if (xhr.status === 200 || xhr.status === 201 || xhr.status === 304) {
                response = loadResponse(xhr);

                // start http GET DCC workaround 'ping' hack
                if (response.status === 'success') {
                    _doGetMethodPing(function(message) {
                        callbackWrapper(message);
                    });
                } else {
                    callbackWrapper('DCC XHR response error');
                }
            } else {
                callbackWrapper('DCC XHR status error');
            }
        };
        xhr.onerror = function() {
            traceLog('submitToDcc() -> XHR Auth error: status=' + xhr.status + ', statusText=' + xhr.statusText);
            state.isLoginBusy = false;
        };

        function _extractFormData(requiredKeys) {
            traceLog('dccAuthenticationUtil::submitToDcc()::_extractFormData() -> entering');
            var formData = {};
            var myFormElm = document.getElementById(opts.loginFormId);

            // make sure all required keys are present.
            var countFormElms = 0;
			countFormElms = _fillFormDataFromElem(requiredKeys, myFormElm, formData);

            if (countFormElms !== requiredKeys.length) {
                console.log('dccAuthenticationUtil::submitToDcc()::_extractFormData() -> ERROR: form data structure, missing required element.');
				return false;
            }

			_fillFormDataFromElem(opts.optionalFormKeys, myFormElm, formData);

            formData['successurl'] = opts.dccHost + opts.successUrl;
            return formData;
        }

		function _fillFormDataFromElem(formKeys, myFormElm, formData) {
            var countFormElms = 0;
            for (i = 0; i < formKeys.length; i++) {
                for (j = 0; j < myFormElm.length; j++) {
                    var elm = myFormElm[j];
                    if (elm.name === formKeys[i]) {
                        traceLog('dccAuthenticationUtil::submitToDcc()::_fillFormDataFromElem() -> adding elm ' + elm.name + '=' + elm.value);
                        formData[formKeys[i]] = elm.value;
                        ++countFormElms;
                    }
                }
            }
			return countFormElms;
		}
		
        var formData = _extractFormData(['username', 'password', 'type']);

        if (!formData) {
            return false;
        }

        var urlEncodedFormData = urlEncodeFormData(formData);
        traceLog('dccAuthenticationUtil::submitToDcc() -> Formdata - ' + urlEncodedFormData);
        xhr.open('post', authLink, true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
        xhr.withCredentials = true;
        xhr.send(urlEncodedFormData);
        traceLog('dccAuthenticationUtil::submitToDcc() -> exiting, returning false');
        return false;
    }

    function urlEncodeFormData(formData) {
        var s = '';
  
        function encode(s) {
            return encodeURIComponent(s).replace(/%20/g, '+');
        }
        
        // http://droidscript.org/javascript/Global_Objects/encodeURIComponent.html
        function fixedEncodeURIComponent(str) {
        	  return encodeURIComponent(str).replace(/[!'()*]/g, function(c) {
        	    return '%' + c.charCodeAt(0).toString(16);
        	  });
       	}

        for (var key in formData) {
            if (formData.hasOwnProperty(key)) {
            	if(key=="password")
                    s += (s ? '&' : '') + key + '=' + fixedEncodeURIComponent(formData[key]);
            	else
            		s += (s ? '&' : '') + encode(key) + '=' + formData[key];
            }
        }

        traceLog('urlEncodeFormData() -> returning ' + s);
        return s;
    }
    
    function loadResponse(xhr) {
        var parsedResponse;

        try {
            parsedResponse = JSON.parse(xhr.responseText);
        } catch (e) {
            console.log('dccAuthenticationUtil::loadResponse() -> ERROR: unable to parse JSON data : ' + e);
            parsedResponse = {
                status: 'error',
                errorCode: 'SERVER_ERROR',
                errorMsg: {
                    fr: 'Une erreur est survenue lors du traitement de votre requête.',
                    en: 'An error occurred while processing your request.'
                },
                errorDtl: {
                    fr: 'Veuillez contacter le soutien technique de Vidéotron.',
                    en: 'Please contact Vidéotron technical support'
                }
            };
        }

        return parsedResponse;
    }

    window.dccAuthenticationUtil = {
        init: init,
        submitToDcc : function () {
            traceLog('dccAuthenticationUtil::submitToDcc() called programmatically');
        	submitToDccWrapper(null);
        }
    };

    function submitToDccWrapper(event) {
    	if (event) {
    		if (event.preventDefault) {
    			event.preventDefault();
    		} else {
    			event.returnValue = false;
    		}
    	}
        if (state.isLoginBusy) {
            traceLog(
                'dccAuthenticationUtil::submitToDccWrapper() -> BLOCKED LOGIN state.isLoginBusy is true (waiting for callback), cancel submit.'
            );
            return false;
        }
        traceLog('dccAuthenticationUtil::submitToDccWrapper() -> isLoginBusy=true');
        state.isLoginBusy = true;
        return submitToDcc(document.getElementById(opts.loginFormId));
    }

    function useAttachEvent() {
        return !(!document.attachEvent || typeof document.attachEvent === 'undefined');
    }

    function dccOnReady(event) {
    	if (!state.init) {
    		state.callOnReadyWhenInit = true;
            traceLog('dccAuthenticationUtil::DOMContentLoaded -> Not yet initialized, postponing event after initialization');
    		return
    	}
        traceLog('dccAuthenticationUtil::DOMContentLoaded -> ENTERING');

        if (typeof opts.loginFormId === 'string') {
            var loginForm = document.getElementById(opts.loginFormId);
            if (loginForm.addEventListener) {
                loginForm.addEventListener('submit', submitToDccWrapper, false);
            } else if (loginForm.attachEvent) {
                loginForm.attachEvent('onsubmit', submitToDccWrapper);
            } else {
                console.log('dccAuthenticationUtil::DOMContentLoaded -> ERROR: form data structure -> unknown login form ID.');
            }
        }

        traceLog('dccAuthenticationUtil::DOMContentLoaded -> calling onReady()...');
        if (opts.onReady) {
            opts.onReady();
        } else {
            traceLog('dccAuthenticationUtil::DOMContentLoaded -> onReady() callback is undefined!');
        }
    }
    
   function IsJsonString(str) {
	    try {
	        JSON.parse(str);
	    } catch (e) {
	        return false;
	    }
	    return true;
	}
   
   function detectIE() {
	    var ua = window.navigator.userAgent;

	    var msie = ua.indexOf('MSIE ');
	    if (msie > 0) {
	        // IE 10 or older => return version number
	        return parseInt(ua.substring(msie + 5, ua.indexOf('.', msie)), 10);
	    }

	    var trident = ua.indexOf('Trident/');
	    if (trident > 0) {
	        // IE 11 => return version number
	        var rv = ua.indexOf('rv:');
	        return parseInt(ua.substring(rv + 3, ua.indexOf('.', rv)), 10);
	    }

	    var edge = ua.indexOf('Edge/');
	    if (edge > 0) {
	       // Edge (IE 12+) => return version number
	       return parseInt(ua.substring(edge + 5, ua.indexOf('.', edge)), 10);
	    }
	    
	    // other browser
	    return false;
	}

    function getLoginStatus(){
    	
    	var isIE = detectIE();
    	if(!isIE || isIE > 11){
			var xmlhttpDccAuth = new XMLHttpRequest(),
			 	url = opts.dccHost + opts.successUrl;
			
			xmlhttpDccAuth.withCredentials = true;
			xmlhttpDccAuth.onreadystatechange = function() {
			    if (xmlhttpDccAuth.readyState == 4) {
			       if (xmlhttpDccAuth.status == 200) {
			    	   if(IsJsonString(xmlhttpDccAuth.responseText)){
			        	   var data = JSON.parse(xmlhttpDccAuth.responseText);
			        	   if(data.status === "success" && data.userId !== "Anonymous" ){
			        		   window.location.reload(true);
			        	   }
			    	   }
			       }
			    }
			};
			
			xmlhttpDccAuth.open("GET", url, true);
			xmlhttpDccAuth.send();
    	}
    }

    if (useAttachEvent()) {
        document.attachEvent('onreadystatechange', function() {
            if (document.readyState === 'complete') {
                return dccOnReady();
            }
        });
    } else {
        document.addEventListener('DOMContentLoaded', dccOnReady);
    }
})(window, document);
